import { AjaxInfo } 							from "./Types/Types";
import { ValidationService }					from "./Services/ValidationService";
import { TabService }						    from "./Services/TabService";
import { ParsleyService }						from "./Services/ParsleyService";
import { ZipAutocompleteService }               from "./Services/ZipAutocompleteService";
import { AddressAutocompleteService }           from "./Services/AddressAutocompleteService";
import { Accordion }                            from "./Components/Accordion";
import { FieldPersistenceService }              from "./Services/FieldPersistenceService";
import { LoginForm }                            from "./Components/LoginForm";
import { Form }                                 from "./Components/Form";
import { FormField }                            from "./Components/FormField";
import { Coupons }                              from "./Components/Coupons";
import { TermsAndConditions }                   from "./Components/TermsAndConditions";
import { PlaceOrderButton }                     from "./Components/PlaceOrderButton";
import { PaymentRequestButtons }                from "./Components/PaymentRequestButtons";
import { CompleteOrderService }                 from "./Services/CompleteOrderService";
import { UpdateCheckoutService }                from "./Services/UpdateCheckoutService";
import { AlertService }                         from "./Services/AlertService";
import { PaymentGatewaysService }               from "./Services/PaymentGatewaysService";
import {Cart} from "./Components/Cart";
import {CountryRowResizeService} from "./Services/CountryRowResizeService";

declare let jQuery: any;
declare let cfwEventData: any;

/**
 * The main class of the front end checkout system
 */
export class Main {
	/**
	 * @type {any}
	 * @private
	 */
	private _checkoutForm: any;

	/**
	 * @type {any}
	 * @private
	 */
	private _tabContainer: any;

    /**
	 * @type {any}
	 * @private
     */
    private _alertContainer: any;

	/**
	 * @type {AjaxInfo}
	 * @private
	 */
	private _ajaxInfo: AjaxInfo;

	/**
	 * @type {any}
	 * @private
	 */
	private _settings: any;

	/**
	 * @type {TabService}
	 * @private
	 */
	private _tabService: TabService;

	/**
	 * @type {UpdateCheckoutService}
	 * @private
	 */
	private _updateCheckoutService: UpdateCheckoutService;

	/**
	 * @type {boolean}
	 * @private
	 */
	private _preserve_alerts: boolean;

	/**
	 * @type any
	 * @private
	 */
	private _updateCheckoutTimer: any;

	/**
	 * @type boolean
	 * @private
	 */
	private _loadTabs: any;

	/**
	 * @type {Main}
	 * @private
	 * @static
	 */
	private static _instance: Main;

	/**
	 * @type {ParsleyService}
	 * @private
	 */
	private _parsleyService: ParsleyService;

	/**
	 * @param {any} checkoutFormElement
	 * @param {any} alertContainer
	 * @param {any} tabContainerElement
	 * @param {any} breadCrumbElement
	 * @param {AjaxInfo} ajaxInfo
	 * @param {any} settings
	 */
	constructor( checkoutFormElement: any, alertContainer: any, tabContainerElement, breadCrumbElement, ajaxInfo: AjaxInfo, settings: any ) {
		Main.instance = this;

		this.checkoutForm               = checkoutFormElement;
		this.tabContainer               = tabContainerElement;
		this.alertContainer             = alertContainer;
		this.ajaxInfo                   = ajaxInfo;
		this.settings                   = settings;
		this.loadTabs                   = this.settings.load_tabs;

		/**
		 * Services
		 */
		// Maybe Load Tab Service
		if ( this.loadTabs ) {
			this.tabService             = new TabService( this.tabContainer, breadCrumbElement );
		}

		// Setup the validation service - has to happen after tabs are setup
		new ValidationService( this.tabService.tabContainer );

		// Field Persistence Service
		new FieldPersistenceService( checkoutFormElement );

		// Parsley Service
		this.parsleyService = new ParsleyService();

		// Zip Autocomplete Service
		new ZipAutocompleteService();

		// Address Autocomplete Service
		new AddressAutocompleteService();

		// Complete Order Service
		new CompleteOrderService();

		// Update Checkout Service
		this.updateCheckoutService = new UpdateCheckoutService();

		// Alert Service
		new AlertService( this.alertContainer );

		// Payment Gateway Service
		new PaymentGatewaysService();

		// Country Row Resize Service
		new CountryRowResizeService();

		/**
		 * Components
		 */
		// Load Form component
		new Form();

		// Load Accordion component
		new Accordion();

		// Load Login Form component
		new LoginForm();

		// Load FormField Component
		new FormField();

		// Load Coupons component
		new Coupons();

		// Load Terms and Conditions Component
		new TermsAndConditions();

		// Load Place Order Button Component
		new PlaceOrderButton();

		// Load Payment Request Buttons Component
		new PaymentRequestButtons();

		// Cart Component
		new Cart();

		/**
		 * Compatibility Classes
		 */
		this.loadCompatibilityClasses();

		jQuery( document.body ).on( 'cfw-remove-overlay', () => {
			Main.removeOverlay();
		} );

		// Page load actions
		jQuery( window ).on( 'load', () => {
			// Give plugins a chance to react to our hidden, invisible shim checkbox
			jQuery( '#ship-to-different-address-checkbox' ).trigger( 'change' );

			/**
			 * On first load, we force updated_checkout to run for gateways
			 * that need it / want it / gotta have it
			 */
			this.updateCheckoutService.force_updated_checkout = true;

			// Don't blow away pre-existing alerts on the first update checkout call
			this.preserve_alerts = true;

			// Trigger initial update checkout
			this.updateCheckoutService.triggerUpdateCheckout();

			// Init checkout ( WooCommerce native event )
			jQuery( document.body ).trigger( 'init_checkout' );
		});
	}

	/**
	 * @param context
	 */
	getFormObject( context: string = 'complete_order' ) {
		let checkout_form: any = this.checkoutForm;
		let bill_to_different_address = <string>jQuery( '[name="bill_to_different_address"]:checked' ).val();
		let $required_inputs = checkout_form.find( '.address-field.validate-required:visible' );
		let has_full_address: boolean = true;
		let lookFor: Array<string> = this.settings.default_address_fields;

		let formData = {
			post_data: checkout_form.serialize()
		};

		if ( $required_inputs.length ) {
			$required_inputs.each( function() {
				if ( jQuery( this ).find( ':input' ).val() === '' ) {
					has_full_address = false;
				}
			});
		}

		let formArr: Array<Object> = checkout_form.serializeArray();
		formArr.forEach(( item: any ) => formData[item.name] = item.value );

		// Handle shipped subscriptions since they are render outside of the form
		jQuery( '#cfw-other-totals input[name^="shipping_method"][type="radio"]:checked, #cfw-other-totals input[name^="shipping_method"][type="hidden"]' ).each(( index, el ) => {
			formData[ jQuery( el ).attr( 'name' ) ] = jQuery( el ).val();
		});

		formData['has_full_address'] = has_full_address;
		formData['bill_to_different_address'] = bill_to_different_address;

		if( bill_to_different_address === 'same_as_shipping' ) {
			lookFor.forEach( field => {
				if( jQuery(`#billing_${field}`).length > 0) {
					formData[`billing_${field}`] = formData[`shipping_${field}`];

					// Make sure the post_data has the same info
					formData['post_data'] = formData['post_data'] + `&billing_${field}=` + formData[`shipping_${field}`];
				}
			});
		}

		if ( context !== 'complete_order' ) {
			// Delete data that we don't need unless we are completing the order
			// TODO: Switch to enumerated data like core uses
			delete formData['woocommerce-process-checkout-nonce'];

			for ( let key in formData ) {
				if ( key.includes('credit-card') || key.includes('token') ) {
					delete formData[ key ];
				}
			}
		}

		return formData;
	}



	/**
	 * Load contextually relevant compatibility classes
	 */
	loadCompatibilityClasses(): void {
		// Compatibility Class Creation
		Object.keys( cfwEventData.compatibility ).forEach( function (key) {
			new (<any>window).CompatibilityClasses[ cfwEventData.compatibility[ key ].class ]( Main.instance, cfwEventData.compatibility[ key ].params ).load( Main.instance, cfwEventData.compatibility[ key ].params );
		} );
	}

	/**
	 * Adds a visual indicator that the checkout is doing something
	 */
	static addOverlay(): void {
		if( jQuery( '#cfw-payment-method:visible' ).length > 0) {
			let main = Main.instance;
			let form = main.checkoutForm;
			let form_data = form.data();

			if ( 1 !== form_data['blockUI.isBlocked'] ) {
				form.block({
					message: null,
					overlayCSS: {
						background: '#fff',
						opacity: 0.6
					}
				});
			}
		}
	}

	/**
	 * Remove the visual indicator
	 */
	static removeOverlay(): void {
		let main = Main.instance;
		let form = main.checkoutForm;

		form.unblock();
	}

	/**
	 * @returns {any}
	 */
	get checkoutForm(): any {
		return this._checkoutForm;
	}

	/**
	 * @param {any} value
	 */
	set checkoutForm( value: any ) {
		this._checkoutForm = value;
	}

	/**
	 * @returns {TabContainer}
	 */
	get tabContainer() {
		return this._tabContainer;
	}

    /**
	 * @return {any}
     */
    get alertContainer(): any {
        return this._alertContainer;
    }

    /**
     * @param {any} value
     */
    set alertContainer( value: any ) {
        this._alertContainer = value;
    }

    /**
	 * @param value
	 */
	set tabContainer( value: any ) {
		this._tabContainer = value;
	}

	/**
	 * @returns {AjaxInfo}
	 */
	get ajaxInfo(): AjaxInfo {
		return this._ajaxInfo;
	}

	/**
	 * @param value
	 */
	set ajaxInfo( value: AjaxInfo ) {
		this._ajaxInfo = value;
	}

	/**
	 * @returns {any}
	 */
	get settings(): any {
		return this._settings;
	}

	/**
	 * @param value
	 */
	set settings( value: any ) {
		this._settings = value;
	}

	/**
	 * @returns {TabService}
	 */
	get tabService(): TabService {
		return this._tabService;
	}

	/**
	 * @param {TabService} value
	 */
	set tabService(value: TabService ) {
		this._tabService = value;
	}

	/**
	 * @returns {UpdateCheckoutService}
	 */
	get updateCheckoutService(): UpdateCheckoutService {
		return this._updateCheckoutService;
	}

	/**
	 * @param {UpdateCheckoutService} value
	 */
	set updateCheckoutService( value: UpdateCheckoutService ) {
		this._updateCheckoutService = value;
	}

	/**
	 * @returns {ParsleyService}
	 */
	get parsleyService(): ParsleyService {
		return this._parsleyService;
	}

	/**
	 * @param {ParsleyService} value
	 */
	set parsleyService(value: ParsleyService) {
		this._parsleyService = value;
	}

	/**
	 * @returns {boolean}
	 */
	get preserve_alerts(): boolean {
		return this._preserve_alerts;
	}

	/**
	 * @param {boolean} value
	 */
	set preserve_alerts( value: boolean ) {
		this._preserve_alerts = value;
	}

	get updateCheckoutTimer(): any {
		return this._updateCheckoutTimer;
	}

	set updateCheckoutTimer( value: any ) {
		this._updateCheckoutTimer = value;
	}

	get loadTabs(): any {
		return this._loadTabs;
	}

	set loadTabs(value: any) {
		this._loadTabs = value;
	}

	/**
	 * @returns {Main}
	 */
	static get instance(): Main {
		return Main._instance;
	}

	/**
	 * @param {Main} value
	 */
	static set instance( value: Main ) {
		if(!Main._instance ) {
			Main._instance = value;
		}
	}
}